// WebSocket
var ws = null;
var nextCommandNumber = 1;
var tracing = false;
var emulatorPaused = true;
var bytesForSkip = 0;				// Number of bytes to skip on a "skip" command
var consoleNeedsTimestamp = true;

var pastInstructions = new Array;
var futureInstructions = new Array;

var lastRegisters = null;		// Saved copy of registers message so we can detect changes to highlight

//====================================================================================================
//
// Configurable parameters
//
//====================================================================================================

var maxPastInstructions = 100;	// Maxiumum number of previous instructions to retain
var maxFutureInstructions = 300;
var currentInstructionOffset = 8;	// How many rows down to put the current instruction

var maxStackDepth = 256;				// Maximum number of bytes worth of stack to show
var stackItems = null;

//====================================================================================================
//
// End configurable parameters
//
//====================================================================================================

jQuery.extend(verge);

// When the page is loaded...

window.addEventListener('load', function(e) {

	// Build the layout grid
	
	var pstyle = 'background-color: #f5f6f7; border: 1px solid #dfdfdf; padding: 5px; overflow: hidden;';
	
	// Processor status flags
	
	var psrbox =	'<table class="psrbox"><tr><th>N</th><th>V</th><th>m</th><th>x</th><th>D</th><th>I</th><th>Z</th><th>C</th></tr>' +
								'<tr>' +
								'<td id="pN">&nbsp;</td><td id="pV">&nbsp;</td><td id="pm">&nbsp;</td><td id="px">&nbsp;</td>' +
								'<td id="pD">&nbsp;</td><td id="pI">&nbsp;</td><td id="pZ">&nbsp;</td><td id="pC">&nbsp;</td></tr>' +
								'</table><div class="regSpacer"></div>&nbsp;&nbsp;<div id="emulationMode"></div>';
	
	// Register value boxes
	
	var regstr = 	'<div class="regName">PBR:&nbsp;</div><div class="regField wide2" id="regPBR">$00</div><div class="regSpacer"></div>' +
								'<div class="regName">PC:&nbsp;</div><div class="regField wide4" id="regPC">$0000</div><div class="regSpacer"></div>' +
								'<div class="regName">DBR:&nbsp;</div><div class="regField wide2" id="regDBR">$00</div><div class="regSpacer"></div>' +
								'<div class="regName">A:&nbsp;</div><div class="regField wide4" id="regA">$0000</div><div class="regSpacer"></div>'+
								'<div class="regName">X:&nbsp;</div><div class="regField wide4" id="regX">$0000</div><div class="regSpacer"></div>' +
								'<div class="regName">Y:&nbsp;</div><div class="regField wide4" id="regY">$0000</div><div class="regSpacer"></div>' +
								'<div class="regName">DP:&nbsp;</div><div class="regField wide4" id="regDP">$0000</div><div class="regSpacer"></div>' +
								'<div class="regName">SP:&nbsp;</div><div class="regField wide4" id="regSP">$0000</div><div class="regSpacer"></div>' +
								'<div class="regName">P:&nbsp;</div><div class="regField wide2" id="regPSR">$00</div><div class="regSpacer"></div>';
									
	$('#layout').w2layout({
		name: "debuglayout",
		panels: [
			{ type: "top", size: 40, resizable: false, style: pstyle, content: "<div id='register-box'>" + regstr + "</div>" + psrbox + "<div id='toolbar'></div>" },
			{ type: "left", size: 200, resizable: true, style: pstyle, content: "<div id='breakpoint-box'></div>" },
			{ type: "right", size: 160, resizable: true, style: pstyle, content: "<div id='stack-box'></div>" },
			{ type: "main", size: "35%", resizable: true, style: pstyle, content: "<div id='console-box'></div>" },
			{ type: "preview", size: "65%", resizable: true, style: pstyle, content: "<div id='code-box'></div>" }
		]
	});
	
	// Build the main toolbar
	
	$('#toolbar').w2toolbar({
		name: 'toolbar',
		items: [
			{ type: 'spacer' },
			{ type: 'button', id: 'emulatorStateButton', caption: "Pause", icon: 'icon-pause', hint: 'Pause the emulator.' }
		],
		onClick: handleToolbarClick
	});
	
	// Build the stack box

	var stackStyle = 'background-color: white; border: 1px solid #dfdfdf; padding: 0px;';

	$('#stack-box').w2layout({
		name: "stack-box",
		panels: [
			{ type: "main", size: "100%", resizable: false, style: stackStyle, content: "<div id='stack-header'>Stack</div><div id='stack-list'></div>" }
		]
	});

	// Build the breakpoints box
	
	$('#breakpoint-box').w2layout({
		name: "breakpoint-box",
		panels: [
			{ type: "main", resizable: false, style: pstyle, content: "<div id='breakpoint-list'></div>" },
			{ type: "bottom", size: 40, resizable: false, style: pstyle, content: "<div id='breakpoint-toolbar'></div>"}
		]
	});
	
	$('#breakpoint-list').w2grid({
		name: 'breakpoint-list',
		size: '100%',
		header: "Breakpoints",
		show: {
			header: true
		},
		columns: [
			{ field: "address", caption: "Address", size: "40%" },
			{ field: "name", caption: "Name", size: "60%" }
		]
	});
	
	$('#breakpoint-toolbar').w2toolbar({
		name: 'breakpoint-toolbar',
		items: [
			{ type: 'button', id: 'addBreakpointButton', icon: 'icon-plus-sign', hint: 'Add a new breakpoint.'},
			{ type: 'button', id: 'removeBreakpointButton', icon: 'icon-minus-sign', hint: 'Remove the selected breakpoint.' },
			{ type: 'spacer' },
			{ type: 'check', id: 'toggleBreakpointsButton', caption: 'Enabled', icon: 'icon-bug', hint: 'Enable breakpoints.' }
		],
		onClick: handleBreakpointBar
	});
	
	// Build the code box
	
	$('#code-box').w2layout({
		name: 'code-box',
		panels: [
			{ type: "main", style: pstyle, content: "<div id='code-list'></div>" },
			{ type: "bottom", size: 40, resizable: false, style: pstyle, content: "<div id='code-toolbar'></div>" }
		]
	});
	
	$('#code-toolbar').w2toolbar({
		name: 'code-toolbar',
		items: [
			{ type: 'button', id: 'stepInButton', caption: 'Step In', icon: 'icon-step-forward', hint: 'Step once, entering subroutines if applicable.'},
			{ type: 'button', id: 'skipButton', caption: 'Skip', icon: 'icon-share-alt', hint: 'Skip this instruction.'},
			{ type: 'button', id: 'continueButton', caption: 'Continue', icon: 'icon-play', hint: 'Resume execution of the code.'},
      { type: 'spacer' },
      { type: 'button', id: 'startDebugging', caption: 'Break', icon: 'icon-off', hint: 'Break now and start stepping.'}
		],
		onClick: handleCodeToolbar
	});
		
	// Build the console box
	
	$('#console-box').w2layout({
		name: 'console-box',
		panels: [
			{ type: "top", size: 40, resizable: false, style: pstyle, content: "<div id='console-toolbar'></div>" },
			{ type: "main", style: pstyle, content: "<div id='console'></div>" }
		]
	});
	
	$('#console-toolbar').w2toolbar({
		name: 'console-toolbar',
		items: [
			{ type: 'spacer' },
			{ type: 'button', id: 'clearConsole', caption: "Clear", icon: 'icon-eraser', hint: 'Clear the contents of the console panel.' }		
		],
		onClick: handleConsoleToolbar
	});
	
	ResizeDebugger();             // make it fit
  $('.editable').jinplace();    // Make editable fields editable
	
	window.addEventListener("resize", ResizeDebugger, false);
	
	// Try connecting to localhost
	
	connectToServer("localhost");
}, false);

//
// connectToServer
//
// Connect to the WebSocket server.
//
function connectToServer(specifiedName) {
	// Check if the browser supports WebSockets...
  if ("WebSocket" in window) {
  	var address = "";
  	
  	if (specifiedName != undefined) {
  		address = specifiedName;
  	} else {
  		address = document.getElementById('serverName').value;
  	}
  	
    ws = new WebSocket('ws://' + address + ':6502', "sweet16-debug-protocol");

    // Invoked when there was an error with the connection.
    
    ws.onerror = function(e) {
      console.log('error', e);
    }

    // Invoked when the socket has been opened successfully.
    
    ws.onopen = function(e) {
      //console.log('open', e);
      getEmulatorStatus();
    }

    // Callback invoked when incoming messages arrive. Event `data` attribute
    // holds the string passed.
    
    ws.onmessage = function(e) {
      //console.log("Message: " + e.data);
      var obj = JSON.parse(e.data);
      //console.log('message', obj);
      
      switch(obj.message) {
        case 'emulatorInfo':
          EmulatorInfo(obj);
          break;
        case 'console':
          PrintToConsole(obj);
          break;
        case 'emulatorStatus':
          UpdateEmulatorStatus(obj);
          break;
        case 'instructions':
        	ProcessInstructions(obj);
        	break;
        case 'break':
        	HandleBreak(obj);
        	break;
        case 'registers':
        	HandleRegisters(obj);
        	break;
        case 'stack':
        	HandleStack(obj);
        	break;
      }
    }

    // Invoked when the socket has been closed.
    
    ws.onclose = function(e) {
    	ws = null;
      //console.log('close', e);
    }

  } else {
    // ...seems like the web browser doesn't support WebSockets.
    alert("SweetBug requires WebSocket technology, which your browser does not appear to support. Please try again using Firefox, Safari, or Google Chrome.");
  }
}

//
// handleToolbarClick
//
// Handles clicks on items in the main toolbar.
//
function handleToolbarClick(target, data) {
	switch(target) {
		case 'emulatorStateButton':
			setEmulatorStatus();
			break;
	}
}

//
// saveNewBreakpoint
//
// Called by the addBreakpoint form to create a new breakpoint.
//
function saveNewBreakpoint() {
	var errors = w2ui['breakpointeditor-form'].validate(true);
	if (errors.length) {
		return;								// Do nothing if there are validation errors
	}
	
	// Time to build the message to send to the debug server
	
	var addressField = w2ui['breakpointeditor-form'].get('field_address');
	var nameField = w2ui['breakpointeditor-form'].get('field_name');
	
  var command = {command: "addBreakpoint", order: nextCommandNumber, type: "break"};
  
  // Add the address to the command
  
  var address = parseInt(addressField.el.value, 16);
  command.address = address;
  
  // Add the name (if any) to the command
  
  var nameStr = nameField.el.value;
  if (nameStr.length) {
	  command.name = nameStr;
  }
  
	sendCommand(command);
	
	// Add it to the breakpoint list
	
	w2ui['breakpoint-list'].add({
		address: MakeAddressString(address),
		name: nameStr,
		recid: w2ui['breakpoint-list'].total + 1
	});
	
	$().w2popup('close');
}

//
// runBreakpointForm
//
// Present the breakpoint editor. Clicking the save button calls
// the routine specified as an input parameter.
//
function runBreakpointForm(saveFunc) {
	$().w2form({
		name: 'breakpointeditor-form',
		style: 'border: 0px; background-color: transparent;',
		formHTML:
			'<div class="w2ui-page page-0">' +
			'  <p>Enter the hex address of the breakpoint and an optional name for the breakpoint.</p><br>' +
			'  <div class="w2ui-label">Address: </div>' +
			'  <div class="w2ui-field">' +
			'    <input name="field_address" type="text" maxlength="6" size="6" />' +
			'  </div>' +
			'  <br>' +
			'  <div class="w2ui-label">Name: </div>' +
			'  <div class="w2ui-field">' +
			'    <input name="field_name" type="text" maxlength="63" size="40" />' +
			'  </div>' +
			'</div>' +
			'<div class="w2ui-buttons">' +
			'  <input type="button" value="Cancel" name="cancel"> <input type="button" value="Save" name="save">' +
			'</div>',
		fields: [
			{ name: 'field_address', type: 'hex', required: true },
			{ name: 'field_name', type: 'text', required: false }
		],
		actions: {
			"cancel": function () { $().w2popup('close'); },			// Cancel always works the same way
			"save": saveFunc
		}
	});
	
	$().w2popup('open', {
		title: 'Add/Edit Breakpoint',
		body: '<div id="breakpointEditor" class="dialogform"></div>',
		style: 'padding: 15px 0px 0px 0px',
		width: 450,
		height: 260,
		onOpen: function() {
			$('#w2ui-popup #breakpointEditor').w2render('breakpointeditor-form');
		}
	});
}

//
// addBreakpoint
//
// Displays the breakpoint editor and lets the user create a new breakpoint.
//
function addBreakpoint() {
	runBreakpointForm(saveNewBreakpoint);
}

//
// removeSelectedBreakpoints
//
// Removes all the selected breakpoints.
//
function removeSelectedBreakpoints() {
	var sel = w2ui['breakpoint-list'].getSelection();
	
	if (sel.length) {
		for (i=0; i<sel.length; i++) {
			var rec = w2ui['breakpoint-list'].get(sel[i]);
			
			// Send a message to the debug server to remove the breakpoint
			
			var command = {command: "clearBreakpoint", order: nextCommandNumber, address:AddressStringToNumber(rec.address)};
			sendCommand(command);
			
			// Now remove it from the UI
			
			w2ui['breakpoint-list'].remove(sel[i]);
		}
	}
}

//
// toggleBreakpoints
//
// Toggles breakpoints on and off
//
function toggleBreakpoints() {
  if (w2ui['breakpoint-toolbar'].get('toggleBreakpointsButton').checked) {
  	w2ui['breakpoint-toolbar'].uncheck('toggleBreakpointsButton');
  } else {
  	w2ui['breakpoint-toolbar'].check('toggleBreakpointsButton');
  }
	setEmulatorStatus();
}

//
// handleBreakpointBar
//
// Handles clicks in the breakpoint toolbar.
//
function handleBreakpointBar(target, data) {
	switch(target) {
		case 'addBreakpointButton':
			addBreakpoint();
			break;
		case 'removeBreakpointButton':
			removeSelectedBreakpoints();
			break;
		case 'toggleBreakpointsButton':
			toggleBreakpoints();
			break;
	}
}

/*
 * Register value changing
 */

//
// runChangeRegisterDialog
//
// Presents a "change register value" dialog and returns the new
// value. Values can be in:
//
//  binary:   lead with a "b"
//  octal:    lead with an "o" or "0"
//  decimal:  no added characters, no leading "0"
//  hex:      lead with "$" or "0x"
//
function runChangeRegisterDialog(regName) {
	$().w2form({
             name: 'changeregister-form',
             style: 'border: 0px; background-color: transparent;',
             formHTML:
             '<div class="w2ui-page page-0">' +
             '  <p>Enter the new value for the ' + regName + ' register.</p><br>' +
             '  <div class="w2ui-label">Value: </div>' +
             '  <div class="w2ui-field">' +
             '    <input name="field_value" type="text" maxlength="17" size="17" />' +
             '  </div>' +
             '</div>' +
             '<div class="w2ui-buttons">' +
             '  <input type="button" value="Cancel" name="cancel"> <input type="button" value="Save" name="save">' +
             '</div>',
             fields: [
                      { name: 'field_value', type: 'clear', required: true }
                      ],
             actions: {
             "cancel": function () { $().w2popup('close'); },			// Cancel always works the same way
             "save": null // SET THIS
             }
  });
	
	$().w2popup('open', {
              title: 'Set value of ' + regName,
              body: '<div id="regValueEditor" class="dialogform"></div>',
              style: 'padding: 15px 0px 0px 0px',
              width: 450,
              height: 200,
              onOpen: function() {
              $('#w2ui-popup #regValueEditor').w2render('changeregister-form');
              }
  });
}

//
// handleCodeToolbar
//
// Handles clicks on the code toolbar.
//
function handleCodeToolbar(target, data) {
	switch(target) {
		case 'stepInButton':
			stepIn();
			break;
		case 'continueButton':
			continueEmulator();
			break;
		case 'skipButton':
			skipInstruction();
			break;
    case 'startDebugging':
      startDebugging();
      break;
	}
}

//
// handleConsoleToolbar
//
// Handles clicks in the console toolbar.
//
function handleConsoleToolbar(target, data) {
	switch(target) {
		case 'clearConsole':
			clearConsole();
			break;
	}
}

//
// sendCommand
//
// Sends the object specified by "command" to the debug server.
//
function sendCommand(command) {
	ws.send(JSON.stringify(command));
	nextCommandNumber++;
}

function clearConsole() {
  var el = document.getElementById("console");
  el.innerHTML = "";
  el.scrollTop = 0;
  consoleNeedsTimestamp = true;
}

function EmulatorInfo(obj) {
  var text = obj.name + " " + obj.version + " (" + obj.build + ")" +
             "\r\n\n" + obj.copyright + "\r\n\n" + obj.date + " at " + obj.time;
  alert(text);
}

function getEmulatorStatus() {
  var command = {command: "getEmulatorStatus", order: nextCommandNumber};
	sendCommand(command);
}

function getEmulatorInfo() {
  var command = {command: "getEmulatorInfo", order: nextCommandNumber};
	sendCommand(command);
}

function stepIn() {
	var command = {command: "step", order: nextCommandNumber, type: "in"};
	sendCommand(command);
}

function startDebugging() {
	var command = {command: "step", order: nextCommandNumber, type: "start"};
	sendCommand(command);
}

function continueEmulator() {

	// Make the top command on the previous instruction stack class lastInstr instead
	// of currentInstr
	
	pastInstructions[pastInstructions.length - 1].extraStyle = ' lastInstr';
	$('.currentInstr').toggleClass('currentInstr lastInstr');

	// Now send the continue command along to the debug server
	var command = {command: "step", order: nextCommandNumber, type: "stop"};
	sendCommand(command);
}

function skipInstruction() {
	var command = {command: "step", order: nextCommandNumber, type: "skip"};
	sendCommand(command);
}

function setTraceStatus() {
  var command;
  var flag;
  
  if (tracing) {
    flag = false;
  } else {
    flag = true;
  }
  
  command = {command: "trace", order: nextCommandNumber,
        enable: flag};
	sendCommand(command);
}

function setEmulatorStatus() {
  var command;
  var btn = w2ui['breakpoint-toolbar'].get('toggleBreakpointsButton');
  
  command = {command: "setEmulatorStatus", order: nextCommandNumber,
        paused: emulatorPaused, breakpointsEnabled: btn.checked };
	sendCommand(command);
}

/******** Handle incoming instructions */

function AppendTextToConsole(text) {
  var el = document.getElementById("console");
  el.innerHTML += text;
  el.scrollTop = el.scrollHeight;
}

//
// ProcessInstructions
//
// Processes incoming "instructions" objects, adding the instructions
// to the appropriate list and refreshing the code listing view.
//
function ProcessInstructions(obj) {
	var order = obj.inReplyTo;
	var type = obj.type;
	
	// If this is a "step" message, it means that the CPU is preparing to execute this
	// instruction. It needs to be pushed onto the "past instructions" stack. We then
	// need to fetch an updated list of future instructions. We won't redraw until those
	// are received.
	
	if (type == "step") {
		var addr = obj.list[0].address;

		pastInstructions.push(obj.list[0]);
		
		// If the stack is full, remove the oldest instruction
		
		if (pastInstructions.length > maxPastInstructions) {
			pastInstructions.shift();
		}
		
		RequestInstructions(addr + obj.list[0].numBytes, maxFutureInstructions);
		return;
	}

	// If this is a "list" message, it's a response to a specific request for instructions.
	// ONCE WE ARE ABLE TO DO SEPARATE CODE LISTS, WE WILL NEED TO ADJUST THIS TO TRACK
	// THE REQUESTING "ORDER" NUMBER.
	//
	// For now, though, we can just replace the contents of the futureInstructions list
	// with these instructions, then request a redraw of the code listing.
	
	if (type != "list") {
		console.log("Unknown instruction message type");
		return;				// no idea what to do with it if it's not a "list" message
	}	
	
	futureInstructions = obj.list;					// Save the list
	
	DrawInstructionList();
}

//
// DrawInstructionList
//
// Draws the instruction list. The instruction that will be executed next
// is at the top of the pastInstructions stack; each earlier item on that
// stack is the previous instruction. The futureInstructions list contains
// upcoming instructions, in order.
//
function DrawInstructionList() {
	var numPastInstructions = pastInstructions.length;
	var listDiv = document.getElementById('code-list');
	var html = "";
	
	// Build HTML for the past instructions first.
	
	for (var i=0; i<numPastInstructions; i++) {
		var instr = pastInstructions[i];
		var addrStr = MakeAddressString(instr.address);
		var cmd = instr.disassembly.substr(23);
		var extraStyle = "";
		
		if (i == numPastInstructions - 1) {
			extraStyle = " currentInstr";
		}
		
		if (instr.extraStyle) {
			extraStyle = extraStyle + instr.extraStyle;
		}
		
		var row = "<div class='coderow" + extraStyle + "'>";
		row += "<div class='codeAddress'>" + addrStr + "</div>";
		row += "<div class='codeInstruction'>" + cmd + "</div>";
		row += "</div>"
		
		html += row;
	}
	
	// Now add the HTML for future instructions.
	
	var numFutureInstructions = futureInstructions.length;
	
	for (var i=0; i<numFutureInstructions; i++) {
		var instr = futureInstructions[i];
		var addrStr = MakeAddressString(instr.address);
		var cmd = instr.disassembly.substr(23);
				
		var row = "<div class='coderow futureInstr'>";
		row += "<div class='codeAddress'>" + addrStr + "</div>";
		row += "<div class='codeInstruction'>" + cmd + "</div>";
		row += "</div>"
		
		html += row;
	}

	listDiv.innerHTML = html;
	
	// Update the scroll position so that the current instruction is visible.
	
	$('#code-list').scrollTop((numPastInstructions - currentInstructionOffset) * 15);
}

function DrawStack() {
	if (stackItems == null) {
		return;
	}
	
	var count = stackItems.length;
	
	// Sort the stack to be sure the items are in order by address
	
	//stackItems.sort();
	
	// Add the items to the stack panel
	
	var el = document.getElementById('stack-list');
	var html = "";
	
	for (var i=0; i<count; i++) {
		var entry = stackItems[i];
		var address = MakeHexString(entry.address, 4);
		var value = MakeHexString(entry.value, entry.size*2);
		
		var row = "<div class='stackrow'>";
		row += "<div class='stackAddress'>" + address + "</div>";
		row += "<div class='stackValue'>" + value + "</div>";
		row += "</div>"
		
		html += row;
	}
	
	el.innerHTML = html;
	
	$('#stack-list').scrollTop(0);
}

/******** Handle received commands */

//
// HandleRegisters
//
// A "registers" message has arrived, updating us with the latest values
// of all the IIgs registers.
//
function HandleRegisters(obj) {
	document.getElementById("regPBR").innerHTML = MakeHexString(obj.PBR, 2);
	document.getElementById("regPC").innerHTML = MakeHexString(obj.PC, 4);
	document.getElementById("regDBR").innerHTML = MakeHexString(obj.DBR, 2);
	document.getElementById("regA").innerHTML = MakeHexString(obj.A, 4);
	document.getElementById("regX").innerHTML = MakeHexString(obj.X, 4);
	document.getElementById("regY").innerHTML = MakeHexString(obj.Y, 4);
	document.getElementById("regDP").innerHTML = MakeHexString(obj.DP, 4);
	document.getElementById("regSP").innerHTML = MakeHexString(obj.SP, 4);
	document.getElementById("regPSR").innerHTML = MakeHexString(obj.PSR, 2);
	
	if (obj.e) {
		document.getElementById("emulationMode").innerHTML = "emulation";
	} else {
		document.getElementById("emulationMode").innerHTML = "native";
	}
	
	// Update the PSR flag boxes
	
	if (obj.PSR & 0x80) {
		$('#pN').addClass('psrBitSet');
	} else {
		$('#pN').removeClass('psrBitSet');
	}
	
	if (obj.PSR & 0x40) {
		$('#pV').addClass('psrBitSet');
	} else {
		$('#pV').removeClass('psrBitSet');
	}
	
	if (obj.PSR & 0x20) {
		$('#pm').addClass('psrBitSet');
	} else {
		$('#pm').removeClass('psrBitSet');
	}
	
	if (obj.PSR & 0x10) {
		$('#px').addClass('psrBitSet');
	} else {
		$('#px').removeClass('psrBitSet');
	}
	
	if (obj.PSR & 0x08) {
		$('#pD').addClass('psrBitSet');
	} else {
		$('#pD').removeClass('psrBitSet');
	}
	
	if (obj.PSR & 0x04) {
		$('#pI').addClass('psrBitSet');
	} else {
		$('#pI').removeClass('psrBitSet');
	}
	
	if (obj.PSR & 0x02) {
		$('#pZ').addClass('psrBitSet');
	} else {
		$('#pZ').removeClass('psrBitSet');
	}
	
	if (obj.PSR & 0x01) {
		$('#pC').addClass('psrBitSet');
	} else {
		$('#pC').removeClass('psrBitSet');
	}

	// Look to see if any have changed and highlight them
	
	if (lastRegisters) {
		$('.regField').removeClass("changed");		// remove the "changed" class from all register fields
	
		if (obj.PBR != lastRegisters.PBR) {
			$('#regPBR').addClass('changed');
		}
		if (obj.PC != lastRegisters.PC) {
			$('#regPC').addClass('changed');
		}
		if (obj.DBR != lastRegisters.DBR) {
			$('#regDBR').addClass('changed');
		}
		if (obj.A != lastRegisters.A) {
			$('#regA').addClass('changed');
		}
		if (obj.X != lastRegisters.X) {
			$('#regX').addClass('changed');
		}
		if (obj.Y != lastRegisters.Y) {
			$('#regY').addClass('changed');
		}
		if (obj.DP != lastRegisters.DP) {
			$('#regPBR').addClass('changed');
		}
		if (obj.SP != lastRegisters.SP) {
			$('#regSP').addClass('changed');
		}
		if (obj.PSR != lastRegisters.PSR) {
			$('#regPSR').addClass('changed');
		}
	}

	lastRegisters = obj;
}

//
// HandleStack
//
// A "stack" message has arrived with updated information about the
// contents of the stack.
//
function HandleStack(obj) {
	stackItems = obj.items;
	DrawStack();
}

//
// HandleBreak
//
// Received when a BRK, ORCA breakpoint, or debug breakpoint is reached.
// by the CPU. We respond by entering stepping mode. The CPU already is,
// so all we need to do is request the first batch of instructions for
// the code display.
//
function HandleBreak(obj) {
	var address = obj.address;
	
	// Request a batch of instructions

	//RequestInstructions(address, 30);		
}

function RequestInstructions(address, count) {
	var command;

	command = {command: "getInstructions", order: nextCommandNumber, address: address, count: count };
	sendCommand(command);
	
	// Ask for the register state too
	
	command = {command: "getRegisters", order: nextCommandNumber};
	sendCommand(command);
	
	// And the stack
	
	command = {command: "getStack", order: nextCommandNumber, numBytes: 256};
	sendCommand(command);
}

function PrintToConsole(obj) {
	var text = "";
  
  if (consoleNeedsTimestamp) {
    var d = new Date(obj.timestamp);
	
    text = "[" + d.toLocaleTimeString() + "] " + obj.text;
    consoleNeedsTimestamp = false;
  } else {
    text = obj.text;
  }
  
  // If the string ends with a newline, we'll want a timestamp for the next output
  
  if (text.substr(-1, 1) == '\n') {
    consoleNeedsTimestamp = true;
  }
  
  text = text.replace(/\n/g, '<br>');
  AppendTextToConsole(text);
}

function UpdateEmulatorStatus(obj) {
  var el = document.getElementById("emulatorState");
  
  // Update settings
  
  emulatorPaused = obj.paused;
  tracing = obj.tracing;
  
  // Update UI to reflect the settings

  if (emulatorPaused) {
  	w2ui['toolbar'].set('emulatorStateButton', { caption: 'Resume', icon: 'icon-play', hint: 'Unpause the emulator.' });
  } else {
    w2ui['toolbar'].set('emulatorStateButton', { caption: 'Pause', icon: 'icon-pause', hint: 'Pause the emulator.' });
  }
  
  if (obj.breakpointsEnabled) {
  	w2ui['breakpoint-toolbar'].check('toggleBreakpointsButton');
  } else {
  	w2ui['breakpoint-toolbar'].uncheck('toggleBreakpointsButton');
  }
  
  /*
  el = document.getElementById("traceStatus");
  if (tracing) {
		el.innerHTML = "Trace: On";
	} else {
		el.innerHTML = "Trace: Off";
  }*/
}

/*
 * Utilities
 */

//
// MakeAddressString
//
// Creates an Apple IIgs address string from the specified integer.
//
function MakeAddressString(address) {
	var str = address.toString(16);
	
	str = str.toUpperCase();				// Now all upper case
	
	// Pad to 6 characters
	
	var len = str.length;
	
	for (i=len; i<6; i++) {
		str = "0" + str;
	}
	
	// Insert the slash and "$"
	
	str = "$" + str.substring(0, 2) + "/" + str.substring(2);
	
	return str;
}

//
// MakeHexString
//
// Converts a number into a hexadecimal string, with leading "$",
// padded with zeroes to be at least "minLength" characters.
//
function MakeHexString(num, minLength) {
	var str = num.toString(16);
	str = str.toUpperCase();
	
	// Pad it
	
	var len = str.length;
	
	for (i=len; i<minLength; i++) {
		str = "0" + str;
	}
	
	str = "$" + str;
	return str;
}

//
// AddressStringToNumber
//
// Convert an address string back to a number.
//
function AddressStringToNumber(str) {
	var ns = str.substring(1,3) + str.substring(4);
	return parseInt(ns, 16);
}

//
// ResizeDebugger
//
// Updates the debugger size to match the window dimensions.
//
function ResizeDebugger() {
/*	var v = $.viewportH() - 150;
	var h = $.viewportW() - 40;
	var el = document.getElementById("layout");
	el.style.width = h + "px";
	el.style.height = v + "px";
	w2ui['debuglayout'].resize();*/
}
